/*
 * Decompiled with CFR 0.152.
 */
package com.nexvor.vectordb;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.nexvor.Nexvor;
import com.nexvor.core.NexvorLogger;
import com.nexvor.vectordb.VectorDBInterface;
import com.nexvor.vectordb.models.SearchResult;
import com.nexvor.vectordb.models.VectorDocument;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

public class PythonVectorDB
implements VectorDBInterface {
    private final Nexvor plugin;
    private final NexvorLogger logger;
    private final Gson gson;
    private Process pythonProcess;
    private BufferedWriter processInput;
    private BufferedReader processOutput;
    private BufferedReader processError;
    private final String pythonExecutable;
    private final Path pythonScriptPath;
    private final Path dbPath;
    private final Path pythonDir;
    private boolean initialized = false;

    public PythonVectorDB(Nexvor plugin) {
        this.plugin = plugin;
        this.logger = plugin.getNexvorLogger();
        this.gson = new Gson();
        this.pythonExecutable = plugin.getConfig().getString("vector_db.python.executable", "python3");
        this.pythonDir = plugin.getDataFolder().toPath().resolve("python");
        this.pythonScriptPath = this.pythonDir.resolve("chroma_server.py");
        this.dbPath = plugin.getDataFolder().toPath().resolve("chromadb");
        try {
            Files.createDirectories(this.pythonDir, new FileAttribute[0]);
            Files.createDirectories(this.dbPath, new FileAttribute[0]);
        }
        catch (IOException e) {
            this.logger.error("Failed to create vector DB directories", e);
        }
    }

    @Override
    public CompletableFuture<Boolean> initialize() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                this.extractPythonResources();
                if (!this.areDependenciesInstalled()) {
                    this.logger.info("Installing Python dependencies (first time only, may take 2-5 minutes)...");
                    if (!this.installDependencies()) {
                        this.logger.severe("Failed to install Python dependencies!");
                        return false;
                    }
                }
                if (!this.startPythonProcess()) {
                    this.logger.severe("Failed to start Python process!");
                    return false;
                }
                JsonObject pingCommand = new JsonObject();
                pingCommand.addProperty("action", "ping");
                CompletableFuture<JsonObject> pingResult = this.sendCommand(pingCommand);
                JsonObject response = pingResult.join();
                if (response != null && response.get("success").getAsBoolean()) {
                    this.initialized = true;
                    this.logger.info("ChromaDB initialized successfully (FREE embeddings active!)");
                    return true;
                }
                this.logger.severe("Failed to ping Python process!");
                return false;
            }
            catch (Exception e) {
                this.logger.error("Failed to initialize PythonVectorDB", e);
                return false;
            }
        });
    }

    private void extractPythonResources() throws IOException {
        this.extractResource("python/chroma_server.py", this.pythonScriptPath);
        this.extractResource("python/requirements.txt", this.pythonDir.resolve("requirements.txt"));
        this.extractResource("python/INSTALL_PYTHON.md", this.pythonDir.resolve("INSTALL_PYTHON.md"));
        this.logger.debug("Extracted Python resources to: " + String.valueOf(this.pythonDir));
    }

    private void extractResource(String resourcePath, Path targetPath) throws IOException {
        try (InputStream in = this.plugin.getResource(resourcePath);){
            if (in == null) {
                throw new IOException("Resource not found: " + resourcePath);
            }
            if (!Files.exists(targetPath, new LinkOption[0])) {
                Files.copy(in, targetPath, new CopyOption[0]);
                this.logger.debug("Extracted: " + resourcePath);
            }
        }
    }

    private boolean areDependenciesInstalled() {
        try {
            ProcessBuilder pb = new ProcessBuilder(this.pythonExecutable, "-c", "import chromadb; import sentence_transformers");
            Process p = pb.start();
            boolean success = p.waitFor(5L, TimeUnit.SECONDS) && p.exitValue() == 0;
            p.destroy();
            return success;
        }
        catch (Exception e) {
            return false;
        }
    }

    private boolean installDependencies() {
        try {
            Path requirementsPath = this.pythonDir.resolve("requirements.txt");
            ProcessBuilder pb = new ProcessBuilder(this.pythonExecutable, "-m", "pip", "install", "-r", requirementsPath.toString());
            pb.redirectErrorStream(true);
            Process p = pb.start();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(p.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    this.logger.info("[pip] " + line);
                }
            }
            boolean success = p.waitFor(300L, TimeUnit.SECONDS) && p.exitValue() == 0;
            p.destroy();
            if (success) {
                this.logger.info("Python dependencies installed successfully!");
            }
            return success;
        }
        catch (Exception e) {
            this.logger.error("Failed to install dependencies", e);
            return false;
        }
    }

    private boolean startPythonProcess() {
        try {
            ProcessBuilder pb = new ProcessBuilder(this.pythonExecutable, this.pythonScriptPath.toString(), this.dbPath.toString());
            pb.redirectErrorStream(false);
            this.pythonProcess = pb.start();
            this.processInput = new BufferedWriter(new OutputStreamWriter(this.pythonProcess.getOutputStream()));
            this.processOutput = new BufferedReader(new InputStreamReader(this.pythonProcess.getInputStream()));
            this.processError = new BufferedReader(new InputStreamReader(this.pythonProcess.getErrorStream()));
            this.startErrorStreamReader();
            Thread.sleep(1000L);
            if (!this.pythonProcess.isAlive()) {
                this.logger.severe("Python process died immediately after start!");
                return false;
            }
            this.logger.info("Python ChromaDB process started (PID: " + this.pythonProcess.pid() + ")");
            return true;
        }
        catch (Exception e) {
            this.logger.error("Failed to start Python process", e);
            return false;
        }
    }

    private void startErrorStreamReader() {
        Thread errorThread = new Thread(() -> {
            try {
                String line;
                while ((line = this.processError.readLine()) != null) {
                    this.logger.debug("[ChromaDB] " + line);
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
        });
        errorThread.setDaemon(true);
        errorThread.setName("ChromaDB-ErrorReader");
        errorThread.start();
    }

    private CompletableFuture<JsonObject> sendCommand(JsonObject command) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (this.pythonProcess == null || !this.pythonProcess.isAlive()) {
                    this.logger.severe("Python process is not running!");
                    return null;
                }
                String commandJson = this.gson.toJson(command);
                this.processInput.write(commandJson);
                this.processInput.newLine();
                this.processInput.flush();
                String responseLine = this.processOutput.readLine();
                if (responseLine == null) {
                    this.logger.severe("No response from Python process!");
                    return null;
                }
                return this.gson.fromJson(responseLine, JsonObject.class);
            }
            catch (IOException e) {
                this.logger.error("Error communicating with Python process", e);
                return null;
            }
        });
    }

    @Override
    public CompletableFuture<Boolean> addDocuments(String collection, List<VectorDocument> documents) {
        JsonObject command = new JsonObject();
        command.addProperty("action", "add");
        command.addProperty("collection", collection);
        ArrayList<String> contents = new ArrayList<String>();
        ArrayList<String> ids = new ArrayList<String>();
        ArrayList<Map<String, Object>> metadatas = new ArrayList<Map<String, Object>>();
        for (VectorDocument doc : documents) {
            contents.add(doc.getContent());
            ids.add(doc.getId());
            metadatas.add(doc.getMetadata());
        }
        command.add("documents", this.gson.toJsonTree(contents));
        command.add("ids", this.gson.toJsonTree(ids));
        command.add("metadatas", this.gson.toJsonTree(metadatas));
        return this.sendCommand(command).thenApply(response -> {
            if (response == null) {
                return false;
            }
            return response.get("success").getAsBoolean();
        });
    }

    @Override
    public CompletableFuture<List<SearchResult>> search(String collection, String query, int limit) {
        JsonObject command = new JsonObject();
        command.addProperty("action", "search");
        command.addProperty("collection", collection);
        command.addProperty("query", query);
        command.addProperty("n_results", limit);
        return this.sendCommand(command).thenApply(response -> {
            ArrayList<SearchResult> results = new ArrayList<SearchResult>();
            if (response == null || !response.get("success").getAsBoolean()) {
                return results;
            }
            JsonArray resultsArray = response.getAsJsonArray("results");
            for (JsonElement resultElement : resultsArray) {
                JsonObject resultObj = resultElement.getAsJsonObject();
                String id = resultObj.get("id").getAsString();
                String document = resultObj.get("document").getAsString();
                double distance = resultObj.get("distance").getAsDouble();
                Map metadata = null;
                if (resultObj.has("metadata") && !resultObj.get("metadata").isJsonNull()) {
                    metadata = this.gson.fromJson(resultObj.get("metadata"), Map.class);
                }
                results.add(new SearchResult(id, document, distance, metadata));
            }
            return results;
        });
    }

    @Override
    public CompletableFuture<Boolean> deleteDocuments(String collection, List<String> ids) {
        JsonObject command = new JsonObject();
        command.addProperty("action", "delete");
        command.addProperty("collection", collection);
        command.add("ids", this.gson.toJsonTree(ids));
        return this.sendCommand(command).thenApply(response -> {
            if (response == null) {
                return false;
            }
            return response.get("success").getAsBoolean();
        });
    }

    @Override
    public CompletableFuture<Integer> getCollectionSize(String collection) {
        JsonObject command = new JsonObject();
        command.addProperty("action", "info");
        command.addProperty("collection", collection);
        return this.sendCommand(command).thenApply(response -> {
            if (response == null || !response.get("success").getAsBoolean()) {
                return 0;
            }
            return response.get("count").getAsInt();
        });
    }

    @Override
    public CompletableFuture<Boolean> clearCollection(String collection) {
        JsonObject command = new JsonObject();
        command.addProperty("action", "clear");
        command.addProperty("collection", collection);
        return this.sendCommand(command).thenApply(response -> {
            if (response == null) {
                return false;
            }
            return response.get("success").getAsBoolean();
        });
    }

    @Override
    public CompletableFuture<Boolean> test() {
        JsonObject command = new JsonObject();
        command.addProperty("action", "ping");
        return this.sendCommand(command).thenApply(response -> {
            if (response == null) {
                return false;
            }
            return response.get("success").getAsBoolean();
        });
    }

    @Override
    public void shutdown() {
        if (this.pythonProcess != null && this.pythonProcess.isAlive()) {
            this.logger.info("Shutting down ChromaDB process...");
            this.pythonProcess.destroy();
            try {
                this.pythonProcess.waitFor(5L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                this.pythonProcess.destroyForcibly();
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean isPythonAvailable(Nexvor plugin) {
        String pythonExec = plugin.getConfig().getString("vector_db.python.executable", "python3");
        try {
            ProcessBuilder pb = new ProcessBuilder(pythonExec, "--version");
            Process p = pb.start();
            boolean success = p.waitFor(5L, TimeUnit.SECONDS);
            int exitCode = p.exitValue();
            p.destroy();
            if (!success) return false;
            if (exitCode != 0) return false;
            pb = new ProcessBuilder(pythonExec, "-c", "import sys; print(f'{sys.version_info.major}.{sys.version_info.minor}')");
            p = pb.start();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(p.getInputStream()));){
                String versionStr = reader.readLine();
                if (versionStr == null) return false;
                String[] parts = versionStr.split("\\.");
                int major = Integer.parseInt(parts[0]);
                int minor = Integer.parseInt(parts[1]);
                if (major >= 3 && minor >= 8) {
                    plugin.getNexvorLogger().info("Found Python " + versionStr);
                    boolean bl = true;
                    return bl;
                }
                plugin.getNexvorLogger().warn("Python " + versionStr + " found, but 3.8+ required");
                boolean bl = false;
                return bl;
            }
        }
        catch (Exception e) {
            return false;
        }
    }
}

