/*
 * Decompiled with CFR 0.152.
 */
package com.nexvor.ratelimit;

import com.nexvor.Nexvor;
import com.nexvor.core.NexvorLogger;
import com.nexvor.ratelimit.RateLimitReason;
import com.nexvor.ratelimit.RateLimitResult;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bukkit.entity.Player;

public class RateLimitManager {
    private final Nexvor plugin;
    private final NexvorLogger logger;
    private final boolean enabled;
    private final int cooldownSeconds;
    private final int maxRequestsPerMinute;
    private final boolean ignoreOps;
    private final Map<UUID, Long> lastRequestTime;
    private final Map<UUID, Queue<Long>> requestHistory;

    public RateLimitManager(Nexvor plugin) {
        this.plugin = plugin;
        this.logger = plugin.getNexvorLogger();
        this.enabled = plugin.getConfig().getBoolean("rate_limiting.enabled", true);
        this.cooldownSeconds = plugin.getConfig().getInt("rate_limiting.cooldown_seconds", 3);
        this.maxRequestsPerMinute = plugin.getConfig().getInt("rate_limiting.max_requests_per_minute", 10);
        this.ignoreOps = plugin.getConfig().getBoolean("rate_limiting.ignore_ops", true);
        this.lastRequestTime = new ConcurrentHashMap<UUID, Long>();
        this.requestHistory = new ConcurrentHashMap<UUID, Queue<Long>>();
        this.logger.info("Rate limiting initialized: " + (this.enabled ? "ENABLED" : "DISABLED") + " (cooldown: " + this.cooldownSeconds + "s, max: " + this.maxRequestsPerMinute + "/min)");
    }

    public RateLimitResult checkAllowed(Player player) {
        long now;
        if (!this.enabled) {
            return RateLimitResult.allowed();
        }
        if (this.ignoreOps && player.isOp()) {
            this.logger.debug("Rate limit bypassed for operator: " + player.getName());
            return RateLimitResult.allowed();
        }
        UUID playerUuid = player.getUniqueId();
        RateLimitResult cooldownResult = this.checkCooldown(playerUuid, now = System.currentTimeMillis());
        if (!cooldownResult.isAllowed()) {
            this.logger.debug("Player " + player.getName() + " rate limited (cooldown): " + cooldownResult.getWaitTimeSeconds() + "s remaining");
            return cooldownResult;
        }
        RateLimitResult windowResult = this.checkSlidingWindow(playerUuid, now);
        if (!windowResult.isAllowed()) {
            this.logger.debug("Player " + player.getName() + " rate limited (too many requests)");
            return windowResult;
        }
        return RateLimitResult.allowed();
    }

    private RateLimitResult checkCooldown(UUID playerUuid, long now) {
        if (!this.lastRequestTime.containsKey(playerUuid)) {
            return RateLimitResult.allowed();
        }
        long lastTime = this.lastRequestTime.get(playerUuid);
        long elapsedSeconds = (now - lastTime) / 1000L;
        if (elapsedSeconds < (long)this.cooldownSeconds) {
            long remainingSeconds;
            String message = "\u00a7cPlease wait \u00a7e" + remainingSeconds + "\u00a7c second" + ((remainingSeconds = (long)this.cooldownSeconds - elapsedSeconds) == 1L ? "" : "s") + " before sending another message.";
            return RateLimitResult.denied(message, RateLimitReason.COOLDOWN, remainingSeconds);
        }
        return RateLimitResult.allowed();
    }

    private RateLimitResult checkSlidingWindow(UUID playerUuid, long now) {
        Queue history = this.requestHistory.computeIfAbsent(playerUuid, k -> new LinkedList());
        long oneMinuteAgo = now - 60000L;
        history.removeIf(time -> time < oneMinuteAgo);
        if (history.size() >= this.maxRequestsPerMinute) {
            String message = "\u00a7cSlow down! You've sent \u00a7e" + this.maxRequestsPerMinute + "\u00a7c messages in the last minute.\n\u00a77Please wait a moment before trying again.";
            return RateLimitResult.denied(message, RateLimitReason.RATE_LIMIT);
        }
        return RateLimitResult.allowed();
    }

    public void recordRequest(Player player) {
        UUID playerUuid = player.getUniqueId();
        long now = System.currentTimeMillis();
        this.lastRequestTime.put(playerUuid, now);
        Queue history = this.requestHistory.computeIfAbsent(playerUuid, k -> new LinkedList());
        history.add(now);
        long oneMinuteAgo = now - 60000L;
        history.removeIf(time -> time < oneMinuteAgo);
        this.logger.debug("Recorded request for " + player.getName() + " (" + history.size() + "/" + this.maxRequestsPerMinute + " this minute)");
    }

    public String getUsageStats(Player player) {
        Long lastTime;
        UUID playerUuid = player.getUniqueId();
        long now = System.currentTimeMillis();
        Queue<Long> history = this.requestHistory.get(playerUuid);
        int requestsThisMinute = 0;
        if (history != null) {
            long oneMinuteAgo = now - 60000L;
            requestsThisMinute = (int)history.stream().filter(time -> time > oneMinuteAgo).count();
        }
        long secondsSinceLastRequest = (lastTime = this.lastRequestTime.get(playerUuid)) != null ? (now - lastTime) / 1000L : 999L;
        return "\u00a77Usage: \u00a7b" + requestsThisMinute + "\u00a77/\u00a7b" + this.maxRequestsPerMinute + "\u00a77 requests this minute | Last request: \u00a7b" + secondsSinceLastRequest + "\u00a77s ago";
    }

    public void cleanup() {
        long now = System.currentTimeMillis();
        long fiveMinutesAgo = now - 300000L;
        int removedCooldowns = 0;
        int removedHistories = 0;
        Iterator<Map.Entry<UUID, Long>> cooldownIterator = this.lastRequestTime.entrySet().iterator();
        while (cooldownIterator.hasNext()) {
            Map.Entry<UUID, Long> entry = cooldownIterator.next();
            if (entry.getValue() >= fiveMinutesAgo) continue;
            cooldownIterator.remove();
            ++removedCooldowns;
        }
        Iterator<Map.Entry<UUID, Queue<Long>>> historyIterator = this.requestHistory.entrySet().iterator();
        while (historyIterator.hasNext()) {
            Map.Entry<UUID, Queue<Long>> entry = historyIterator.next();
            Queue<Long> history = entry.getValue();
            long oneMinuteAgo = now - 60000L;
            history.removeIf(time -> time < oneMinuteAgo);
            if (!history.isEmpty()) continue;
            historyIterator.remove();
            ++removedHistories;
        }
        if (removedCooldowns > 0 || removedHistories > 0) {
            this.logger.debug("Cleaned up rate limit data: " + removedCooldowns + " cooldowns, " + removedHistories + " histories");
        }
    }

    public void clearPlayer(UUID playerUuid) {
        this.lastRequestTime.remove(playerUuid);
        this.requestHistory.remove(playerUuid);
        this.logger.debug("Cleared rate limit data for player: " + String.valueOf(playerUuid));
    }

    public void clearAll() {
        int cooldowns = this.lastRequestTime.size();
        int histories = this.requestHistory.size();
        this.lastRequestTime.clear();
        this.requestHistory.clear();
        this.logger.info("Cleared all rate limit data (" + cooldowns + " cooldowns, " + histories + " histories)");
    }

    public String getStats() {
        return "Rate Limiting Stats:\n  Enabled: " + this.enabled + "\n  Cooldown: " + this.cooldownSeconds + "s\n  Max requests: " + this.maxRequestsPerMinute + "/min\n  Ignore ops: " + this.ignoreOps + "\n  Tracked players (cooldown): " + this.lastRequestTime.size() + "\n  Tracked players (history): " + this.requestHistory.size();
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    public int getCooldownSeconds() {
        return this.cooldownSeconds;
    }

    public int getMaxRequestsPerMinute() {
        return this.maxRequestsPerMinute;
    }

    public boolean isIgnoreOps() {
        return this.ignoreOps;
    }
}

