/*
 * Decompiled with CFR 0.152.
 */
package com.nexvor.rag;

import com.nexvor.Nexvor;
import com.nexvor.core.NexvorLogger;
import com.nexvor.memory.DatabaseManager;
import com.nexvor.rag.DocumentChunker;
import com.nexvor.rag.DocumentParser;
import com.nexvor.rag.models.DocumentChunk;
import com.nexvor.rag.models.KnowledgeBase;
import com.nexvor.vectordb.VectorDBManager;
import com.nexvor.vectordb.models.VectorDocument;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

public class KnowledgeBaseManager {
    private final Nexvor plugin;
    private final NexvorLogger logger;
    private final DatabaseManager databaseManager;
    private final VectorDBManager vectorDBManager;
    private final DocumentParser documentParser;
    private final DocumentChunker documentChunker;
    private final int chunkSize;
    private final int overlapSize;
    private final long maxFileSizeBytes;

    public KnowledgeBaseManager(Nexvor plugin) {
        this.plugin = plugin;
        this.logger = plugin.getNexvorLogger();
        this.databaseManager = plugin.getDatabaseManager();
        this.vectorDBManager = plugin.getVectorDBManager();
        this.chunkSize = plugin.getConfig().getInt("rag.chunking.chunk_size", 800);
        this.overlapSize = plugin.getConfig().getInt("rag.chunking.overlap_size", 100);
        this.maxFileSizeBytes = plugin.getConfig().getLong("rag.max_file_size_mb", 5L) * 1024L * 1024L;
        this.documentParser = new DocumentParser();
        this.documentChunker = new DocumentChunker(this.chunkSize, this.overlapSize);
        this.logger.info("Knowledge base manager initialized (chunk: " + this.chunkSize + ", overlap: " + this.overlapSize + ")");
    }

    public CompletableFuture<Integer> uploadDocument(String kbId, Path filePath, String kbName, String kbDescription) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!Files.exists(filePath, new LinkOption[0])) {
                    this.logger.warn("File not found: " + String.valueOf(filePath));
                    return 0;
                }
                String fileName = filePath.getFileName().toString();
                if (!this.documentParser.isSupported(fileName)) {
                    this.logger.warn("Unsupported file type: " + fileName);
                    return 0;
                }
                if (!this.documentParser.validateFileSize(filePath, this.maxFileSizeBytes)) {
                    this.logger.warn("File too large: " + fileName + " (max: " + this.maxFileSizeBytes / 1024L / 1024L + "MB)");
                    return 0;
                }
                KnowledgeBase kb = this.databaseManager.getKnowledgeBase(kbId);
                if (kb == null) {
                    String collection = this.plugin.getConfig().getString("rag.default_collection", "server_knowledge");
                    kb = new KnowledgeBase(kbId, kbName, kbDescription, collection);
                    this.databaseManager.createKnowledgeBase(kb);
                    this.logger.info("Created new knowledge base: " + kbName + " (ID: " + kbId + ")");
                }
                this.logger.debug("Parsing document: " + fileName);
                String content = this.documentParser.parse(filePath);
                if (content == null || content.isEmpty()) {
                    this.logger.warn("Document is empty: " + fileName);
                    return 0;
                }
                this.logger.debug("Chunking document: " + fileName);
                List<String> chunks = this.documentChunker.chunkSemantic(content);
                this.logger.debug("Created " + chunks.size() + " chunks from " + fileName);
                ArrayList<VectorDocument> vectorDocs = new ArrayList<VectorDocument>();
                ArrayList<DocumentChunk> docChunks = new ArrayList<DocumentChunk>();
                for (int i = 0; i < chunks.size(); ++i) {
                    String chunkId = UUID.randomUUID().toString();
                    String chunkContent = chunks.get(i);
                    DocumentChunk docChunk = new DocumentChunk(chunkId, kbId, fileName, i, chunkContent);
                    docChunks.add(docChunk);
                    VectorDocument vectorDoc = new VectorDocument(chunkId, chunkContent, docChunk.getMetadata());
                    vectorDocs.add(vectorDoc);
                }
                this.logger.debug("Storing chunks in vector DB (collection: " + kb.getCollection() + ")");
                boolean success = this.vectorDBManager.addDocuments(kb.getCollection(), vectorDocs).join();
                if (!success) {
                    this.logger.severe("Failed to store chunks in vector DB");
                    return 0;
                }
                for (DocumentChunk chunk : docChunks) {
                    this.databaseManager.createDocumentChunk(chunk);
                }
                kb.incrementChunkCount(chunks.size());
                this.databaseManager.updateKnowledgeBase(kb);
                this.logger.info("Successfully uploaded " + fileName + " (" + chunks.size() + " chunks) to KB: " + kbName);
                return chunks.size();
            }
            catch (IOException e) {
                this.logger.error("Failed to upload document: " + String.valueOf(filePath), e);
                return 0;
            }
            catch (Exception e) {
                this.logger.error("Unexpected error uploading document", e);
                return 0;
            }
        });
    }

    public List<KnowledgeBase> listKnowledgeBases() {
        return this.databaseManager.getAllKnowledgeBases();
    }

    public KnowledgeBase getKnowledgeBase(String kbId) {
        return this.databaseManager.getKnowledgeBase(kbId);
    }

    public CompletableFuture<Boolean> deleteKnowledgeBase(String kbId) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                KnowledgeBase kb = this.databaseManager.getKnowledgeBase(kbId);
                if (kb == null) {
                    this.logger.warn("Knowledge base not found: " + kbId);
                    return false;
                }
                List<DocumentChunk> chunks = this.databaseManager.getDocumentChunksByKB(kbId);
                ArrayList<String> chunkIds = new ArrayList<String>();
                for (DocumentChunk chunk : chunks) {
                    chunkIds.add(chunk.getId());
                }
                if (!chunkIds.isEmpty()) {
                    this.logger.debug("Deleting " + chunkIds.size() + " chunks from vector DB");
                    boolean success = this.vectorDBManager.deleteDocuments(kb.getCollection(), chunkIds).join();
                    if (!success) {
                        this.logger.warn("Failed to delete some chunks from vector DB");
                    }
                }
                this.databaseManager.deleteDocumentChunksByKB(kbId);
                this.databaseManager.deleteKnowledgeBase(kbId);
                this.logger.info("Deleted knowledge base: " + kb.getName() + " (" + chunks.size() + " chunks)");
                return true;
            }
            catch (Exception e) {
                this.logger.error("Failed to delete knowledge base: " + kbId, e);
                return false;
            }
        });
    }

    public String getStatistics() {
        List<KnowledgeBase> kbs = this.listKnowledgeBases();
        int totalChunks = 0;
        for (KnowledgeBase kb : kbs) {
            totalChunks += kb.getChunkCount();
        }
        return "Knowledge Bases: " + kbs.size() + ", Total Chunks: " + totalChunks;
    }

    public boolean isVectorDBAvailable() {
        return this.vectorDBManager != null && this.vectorDBManager.test().join() != false;
    }

    public DocumentParser getDocumentParser() {
        return this.documentParser;
    }

    public DocumentChunker getDocumentChunker() {
        return this.documentChunker;
    }
}

