/*
 * Decompiled with CFR 0.152.
 */
package com.nexvor.memory;

import com.nexvor.Nexvor;
import com.nexvor.core.NexvorLogger;
import com.nexvor.events.EventType;
import com.nexvor.memory.models.Conversation;
import com.nexvor.memory.models.PlayerData;
import com.nexvor.rag.models.DocumentChunk;
import com.nexvor.rag.models.KnowledgeBase;
import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public class DatabaseManager {
    private final Nexvor plugin;
    private final NexvorLogger logger;
    private final File databaseFile;
    private Connection connection;

    public DatabaseManager(Nexvor plugin) {
        this.plugin = plugin;
        this.logger = plugin.getNexvorLogger();
        this.databaseFile = new File(plugin.getDataFolder(), "nexvor.db");
    }

    public void initialize() {
        try {
            if (!this.plugin.getDataFolder().exists()) {
                this.plugin.getDataFolder().mkdirs();
            }
            Class.forName("org.sqlite.JDBC");
            String url2 = "jdbc:sqlite:" + this.databaseFile.getAbsolutePath();
            this.connection = DriverManager.getConnection(url2);
            this.logger.info("Database connected: " + this.databaseFile.getAbsolutePath());
            this.createTables();
        }
        catch (ClassNotFoundException e) {
            this.logger.severe("SQLite JDBC driver not found! This should not happen.");
            e.printStackTrace();
        }
        catch (SQLException e) {
            this.logger.severe("Failed to initialize database!");
            e.printStackTrace();
        }
    }

    private void createTables() throws SQLException {
        String createPlayersTable = "CREATE TABLE IF NOT EXISTS players (\n    uuid TEXT PRIMARY KEY,\n    username TEXT NOT NULL,\n    first_seen INTEGER NOT NULL,\n    last_seen INTEGER NOT NULL,\n    opted_out INTEGER DEFAULT 0\n)\n";
        String createConversationsTable = "CREATE TABLE IF NOT EXISTS conversations (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    player_uuid TEXT NOT NULL,\n    role TEXT NOT NULL,\n    content TEXT NOT NULL,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
        String createConversationsIndex = "CREATE INDEX IF NOT EXISTS idx_conversations_player_timestamp\nON conversations(player_uuid, timestamp DESC)\n";
        String createGroupsTable = "CREATE TABLE IF NOT EXISTS conversation_groups (\n    group_id TEXT PRIMARY KEY,\n    group_name TEXT NOT NULL,\n    owner_uuid TEXT NOT NULL,\n    created_at INTEGER NOT NULL,\n    active INTEGER DEFAULT 1,\n    FOREIGN KEY (owner_uuid) REFERENCES players(uuid)\n)\n";
        String createGroupMembersTable = "CREATE TABLE IF NOT EXISTS group_members (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    group_id TEXT NOT NULL,\n    player_uuid TEXT NOT NULL,\n    joined_at INTEGER NOT NULL,\n    FOREIGN KEY (group_id) REFERENCES conversation_groups(group_id),\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid),\n    UNIQUE(group_id, player_uuid)\n)\n";
        String createGroupMessagesTable = "CREATE TABLE IF NOT EXISTS group_messages (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    group_id TEXT NOT NULL,\n    player_uuid TEXT,\n    role TEXT NOT NULL,\n    content TEXT NOT NULL,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (group_id) REFERENCES conversation_groups(group_id),\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
        String createGroupMessagesIndex = "CREATE INDEX IF NOT EXISTS idx_group_messages_group_timestamp\nON group_messages(group_id, timestamp DESC)\n";
        String createGlobalMessagesTable = "CREATE TABLE IF NOT EXISTS global_messages (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    conversation_id TEXT NOT NULL,\n    player_uuid TEXT,\n    role TEXT NOT NULL,\n    content TEXT NOT NULL,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
        String createGlobalMessagesIndex = "CREATE INDEX IF NOT EXISTS idx_global_messages_conversation_timestamp\nON global_messages(conversation_id, timestamp DESC)\n";
        String createProximityMessagesTable = "CREATE TABLE IF NOT EXISTS proximity_messages (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    group_id TEXT NOT NULL,\n    player_uuid TEXT,\n    role TEXT NOT NULL,\n    content TEXT NOT NULL,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
        String createProximityMessagesIndex = "CREATE INDEX IF NOT EXISTS idx_proximity_messages_group_timestamp\nON proximity_messages(group_id, timestamp DESC)\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createPlayersTable);
            stmt.execute(createConversationsTable);
            stmt.execute(createConversationsIndex);
            stmt.execute(createGroupsTable);
            stmt.execute(createGroupMembersTable);
            stmt.execute(createGroupMessagesTable);
            stmt.execute(createGroupMessagesIndex);
            stmt.execute(createGlobalMessagesTable);
            stmt.execute(createGlobalMessagesIndex);
            stmt.execute(createProximityMessagesTable);
            stmt.execute(createProximityMessagesIndex);
            String createEventPreferencesTable = "CREATE TABLE IF NOT EXISTS event_preferences (\n    player_uuid TEXT NOT NULL,\n    event_type TEXT NOT NULL,\n    enabled INTEGER NOT NULL DEFAULT 1,\n    PRIMARY KEY (player_uuid, event_type)\n)\n";
            stmt.execute(createEventPreferencesTable);
            String createDiscordLinksTable = "CREATE TABLE IF NOT EXISTS discord_links (\n    minecraft_uuid TEXT PRIMARY KEY,\n    discord_id TEXT NOT NULL UNIQUE,\n    linked_at INTEGER NOT NULL,\n    FOREIGN KEY (minecraft_uuid) REFERENCES players(uuid)\n)\n";
            String createDiscordLinksIndex = "CREATE INDEX IF NOT EXISTS idx_discord_id\nON discord_links(discord_id)\n";
            String createDiscordConversationsTable = "CREATE TABLE IF NOT EXISTS discord_conversations (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    discord_id TEXT NOT NULL,\n    role TEXT NOT NULL,\n    content TEXT NOT NULL,\n    timestamp INTEGER NOT NULL\n)\n";
            String createDiscordConversationsIndex = "CREATE INDEX IF NOT EXISTS idx_discord_conversations\nON discord_conversations(discord_id, timestamp DESC)\n";
            stmt.execute(createDiscordLinksTable);
            stmt.execute(createDiscordLinksIndex);
            stmt.execute(createDiscordConversationsTable);
            stmt.execute(createDiscordConversationsIndex);
            String createCommandLogsTable = "CREATE TABLE IF NOT EXISTS command_logs (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    player_uuid TEXT NOT NULL,\n    player_name TEXT NOT NULL,\n    command TEXT NOT NULL,\n    success INTEGER NOT NULL,\n    dry_run INTEGER NOT NULL,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
            String createCommandLogsIndex = "CREATE INDEX IF NOT EXISTS idx_command_logs_player_timestamp\nON command_logs(player_uuid, timestamp DESC)\n";
            stmt.execute(createCommandLogsTable);
            stmt.execute(createCommandLogsIndex);
            String createModerationLogsTable = "CREATE TABLE IF NOT EXISTS moderation_logs (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    player_uuid TEXT NOT NULL,\n    player_name TEXT NOT NULL,\n    message TEXT NOT NULL,\n    severity TEXT NOT NULL,\n    action TEXT NOT NULL,\n    reason TEXT,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
            String createModerationLogsPlayerIndex = "CREATE INDEX IF NOT EXISTS idx_moderation_logs_player_timestamp\nON moderation_logs(player_uuid, timestamp DESC)\n";
            String createModerationLogsSeverityIndex = "CREATE INDEX IF NOT EXISTS idx_moderation_logs_severity_timestamp\nON moderation_logs(severity, timestamp DESC)\n";
            stmt.execute(createModerationLogsTable);
            stmt.execute(createModerationLogsPlayerIndex);
            stmt.execute(createModerationLogsSeverityIndex);
            try {
                stmt.execute("ALTER TABLE players ADD COLUMN active_bot TEXT DEFAULT 'default'");
                this.logger.info("Added active_bot column to players table");
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            try {
                stmt.execute("ALTER TABLE conversations ADD COLUMN bot_name TEXT DEFAULT 'default'");
                this.logger.info("Added bot_name column to conversations table");
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            try {
                stmt.execute("ALTER TABLE group_messages ADD COLUMN bot_name TEXT DEFAULT 'default'");
                this.logger.info("Added bot_name column to group_messages table");
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            try {
                stmt.execute("ALTER TABLE global_messages ADD COLUMN bot_name TEXT DEFAULT 'default'");
                this.logger.info("Added bot_name column to global_messages table");
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            try {
                stmt.execute("ALTER TABLE proximity_messages ADD COLUMN bot_name TEXT DEFAULT 'default'");
                this.logger.info("Added bot_name column to proximity_messages table");
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            try {
                stmt.execute("ALTER TABLE discord_conversations ADD COLUMN bot_name TEXT DEFAULT 'default'");
                this.logger.info("Added bot_name column to discord_conversations table");
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            try {
                stmt.execute("CREATE INDEX IF NOT EXISTS idx_conversations_bot ON conversations(player_uuid, bot_name, timestamp DESC)");
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            String createKnowledgeBasesTable = "CREATE TABLE IF NOT EXISTS knowledge_bases (\n    id TEXT PRIMARY KEY,\n    name TEXT NOT NULL,\n    description TEXT,\n    collection TEXT NOT NULL,\n    chunk_count INTEGER DEFAULT 0,\n    created_at INTEGER NOT NULL,\n    updated_at INTEGER NOT NULL\n)\n";
            String createDocumentChunksTable = "CREATE TABLE IF NOT EXISTS document_chunks (\n    id TEXT PRIMARY KEY,\n    kb_id TEXT NOT NULL,\n    source_file TEXT NOT NULL,\n    chunk_index INTEGER NOT NULL,\n    content_preview TEXT,\n    created_at INTEGER NOT NULL,\n    FOREIGN KEY (kb_id) REFERENCES knowledge_bases(id) ON DELETE CASCADE\n)\n";
            String createDocumentChunksIndex = "CREATE INDEX IF NOT EXISTS idx_document_chunks_kb\nON document_chunks(kb_id, chunk_index)\n";
            stmt.execute(createKnowledgeBasesTable);
            stmt.execute(createDocumentChunksTable);
            stmt.execute(createDocumentChunksIndex);
            String createDashboardSessionsTable = "CREATE TABLE IF NOT EXISTS dashboard_sessions (\n    id TEXT PRIMARY KEY,\n    user_uuid TEXT NOT NULL,\n    user_name TEXT NOT NULL,\n    session_token TEXT NOT NULL,\n    created_at INTEGER NOT NULL,\n    expires_at INTEGER NOT NULL,\n    last_active INTEGER NOT NULL\n)\n";
            String createAccessCodesTable = "CREATE TABLE IF NOT EXISTS access_codes (\n    code TEXT PRIMARY KEY,\n    player_uuid TEXT NOT NULL,\n    player_name TEXT NOT NULL,\n    created_at INTEGER NOT NULL,\n    expires_at INTEGER NOT NULL,\n    used INTEGER DEFAULT 0\n)\n";
            String createDashboardUsersTable = "CREATE TABLE IF NOT EXISTS dashboard_users (\n    player_uuid TEXT PRIMARY KEY,\n    player_name TEXT NOT NULL,\n    password_hash TEXT NOT NULL,\n    created_at INTEGER NOT NULL,\n    last_login INTEGER\n)\n";
            String createAPITokensTable = "CREATE TABLE IF NOT EXISTS api_tokens (\n    token TEXT PRIMARY KEY,\n    player_uuid TEXT NOT NULL,\n    player_name TEXT NOT NULL,\n    purpose TEXT,\n    scopes TEXT,\n    created_at INTEGER NOT NULL,\n    last_used INTEGER,\n    revoked INTEGER DEFAULT 0\n)\n";
            String createAPITokenLogsTable = "CREATE TABLE IF NOT EXISTS api_token_logs (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    token TEXT NOT NULL,\n    endpoint TEXT NOT NULL,\n    method TEXT NOT NULL,\n    success INTEGER NOT NULL,\n    timestamp INTEGER NOT NULL\n)\n";
            String createDashboardIndexes = "CREATE INDEX IF NOT EXISTS idx_sessions_token\nON dashboard_sessions(session_token)\n";
            String createAccessCodesIndex = "CREATE INDEX IF NOT EXISTS idx_access_codes_player\nON access_codes(player_uuid, expires_at DESC)\n";
            String createAPITokensIndex = "CREATE INDEX IF NOT EXISTS idx_api_tokens_player\nON api_tokens(player_uuid, revoked)\n";
            String createAPITokenLogsIndex = "CREATE INDEX IF NOT EXISTS idx_api_token_logs_token\nON api_token_logs(token, timestamp DESC)\n";
            stmt.execute(createDashboardSessionsTable);
            stmt.execute(createAccessCodesTable);
            stmt.execute(createDashboardUsersTable);
            stmt.execute(createAPITokensTable);
            stmt.execute(createAPITokenLogsTable);
            stmt.execute(createDashboardIndexes);
            stmt.execute(createAccessCodesIndex);
            stmt.execute(createAPITokensIndex);
            stmt.execute(createAPITokenLogsIndex);
            this.logger.info("Database tables created/verified");
        }
    }

    public void close() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                this.connection.close();
                this.logger.info("Database connection closed");
            }
        }
        catch (SQLException e) {
            this.logger.error("Error closing database connection", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public PlayerData getOrCreatePlayer(UUID uuid, String username) {
        try {
            String selectSql = "SELECT * FROM players WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(selectSql);){
                stmt.setString(1, uuid.toString());
                ResultSet rs = stmt.executeQuery();
                if (rs.next()) {
                    this.updatePlayer(uuid, username);
                    PlayerData playerData = new PlayerData(uuid, rs.getString("username"), rs.getLong("first_seen"), rs.getLong("last_seen"), rs.getInt("opted_out") == 1);
                    return playerData;
                }
            }
            long now = System.currentTimeMillis();
            String insertSql = "INSERT INTO players (uuid, username, first_seen, last_seen, opted_out) VALUES (?, ?, ?, ?, 0)";
            try (PreparedStatement stmt2 = this.connection.prepareStatement(insertSql);){
                stmt2.setString(1, uuid.toString());
                stmt2.setString(2, username);
                stmt2.setLong(3, now);
                stmt2.setLong(4, now);
                stmt2.executeUpdate();
            }
            this.logger.debug("Created new player record: " + username + " (" + String.valueOf(uuid) + ")");
            return new PlayerData(uuid, username, now, now, false);
        }
        catch (SQLException e) {
            this.logger.error("Failed to get/create player: " + username, e);
            return null;
        }
    }

    private void updatePlayer(UUID uuid, String username) {
        try {
            String sql = "UPDATE players SET username = ?, last_seen = ? WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, username);
                stmt.setLong(2, System.currentTimeMillis());
                stmt.setString(3, uuid.toString());
                stmt.executeUpdate();
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to update player: " + username, e);
        }
    }

    public void saveMessage(UUID playerUuid, String role, String content) {
        try {
            String sql = "INSERT INTO conversations (player_uuid, role, content, timestamp) VALUES (?, ?, ?, ?)";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                stmt.setString(2, role);
                stmt.setString(3, content);
                stmt.setLong(4, System.currentTimeMillis());
                stmt.executeUpdate();
            }
            this.logger.debug("Saved " + role + " message for player " + String.valueOf(playerUuid));
        }
        catch (SQLException e) {
            this.logger.error("Failed to save conversation message", e);
        }
    }

    public List<Conversation> getConversationHistory(UUID playerUuid, int limit) {
        ArrayList<Conversation> history = new ArrayList<Conversation>();
        try {
            String sql = "SELECT id, player_uuid, role, content, timestamp\nFROM conversations\nWHERE player_uuid = ?\nORDER BY timestamp DESC\nLIMIT ?\n";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                stmt.setInt(2, limit);
                ResultSet rs = stmt.executeQuery();
                while (rs.next()) {
                    history.add(new Conversation(rs.getInt("id"), UUID.fromString(rs.getString("player_uuid")), rs.getString("role"), rs.getString("content"), rs.getLong("timestamp")));
                }
            }
            ArrayList<Conversation> reversed = new ArrayList<Conversation>();
            for (int i = history.size() - 1; i >= 0; --i) {
                reversed.add((Conversation)history.get(i));
            }
            return reversed;
        }
        catch (SQLException e) {
            this.logger.error("Failed to get conversation history", e);
            return new ArrayList<Conversation>();
        }
    }

    public void clearConversationHistory(UUID playerUuid) {
        try {
            String sql = "DELETE FROM conversations WHERE player_uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                int deleted = stmt.executeUpdate();
                this.logger.info("Cleared " + deleted + " messages for player " + String.valueOf(playerUuid));
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to clear conversation history", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int getMessageCount(UUID playerUuid) {
        try {
            String sql = "SELECT COUNT(*) FROM conversations WHERE player_uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                ResultSet rs = stmt.executeQuery();
                if (!rs.next()) return 0;
                int n = rs.getInt(1);
                return n;
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to get message count", e);
        }
        return 0;
    }

    public void setOptOut(UUID playerUuid, boolean optedOut) {
        try {
            String sql = "UPDATE players SET opted_out = ? WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setInt(1, optedOut ? 1 : 0);
                stmt.setString(2, playerUuid.toString());
                stmt.executeUpdate();
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to set opt-out status", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean isOptedOut(UUID playerUuid) {
        try {
            String sql = "SELECT opted_out FROM players WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                ResultSet rs = stmt.executeQuery();
                if (!rs.next()) return false;
                boolean bl = rs.getInt("opted_out") == 1;
                return bl;
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to check opt-out status", e);
        }
        return false;
    }

    public Connection getConnection() {
        return this.connection;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Boolean getEventPreference(UUID playerUuid, EventType eventType) {
        String sql = "SELECT enabled FROM event_preferences WHERE player_uuid = ? AND event_type = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            stmt.setString(2, eventType.getConfigKey());
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return null;
            Boolean bl = rs.getInt("enabled") == 1;
            return bl;
        }
        catch (SQLException e) {
            this.logger.error("Error getting event preference", e);
        }
        return null;
    }

    public void setEventPreference(UUID playerUuid, EventType eventType, boolean enabled) {
        String sql = "INSERT INTO event_preferences (player_uuid, event_type, enabled)\nVALUES (?, ?, ?)\nON CONFLICT(player_uuid, event_type)\nDO UPDATE SET enabled = excluded.enabled\n";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            stmt.setString(2, eventType.getConfigKey());
            stmt.setInt(3, enabled ? 1 : 0);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Error setting event preference", e);
        }
    }

    public Map<String, Boolean> getAllEventPreferences(UUID playerUuid) {
        HashMap<String, Boolean> preferences = new HashMap<String, Boolean>();
        String sql = "SELECT event_type, enabled FROM event_preferences WHERE player_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                preferences.put(rs.getString("event_type"), rs.getInt("enabled") == 1);
            }
        }
        catch (SQLException e) {
            this.logger.error("Error getting all event preferences", e);
        }
        return preferences;
    }

    public void clearEventPreferences(UUID playerUuid) {
        String sql = "DELETE FROM event_preferences WHERE player_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Error clearing event preferences", e);
        }
    }

    public void linkDiscordAccount(UUID minecraftUuid, String discordId) {
        String sql = "INSERT OR REPLACE INTO discord_links (minecraft_uuid, discord_id, linked_at)\nVALUES (?, ?, ?)\n";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, minecraftUuid.toString());
            stmt.setString(2, discordId);
            stmt.setLong(3, System.currentTimeMillis());
            stmt.executeUpdate();
            this.logger.debug("Linked Discord " + discordId + " to Minecraft " + String.valueOf(minecraftUuid));
        }
        catch (SQLException e) {
            this.logger.error("Error linking Discord account", e);
        }
    }

    public void unlinkDiscordAccount(UUID minecraftUuid) {
        String sql = "DELETE FROM discord_links WHERE minecraft_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, minecraftUuid.toString());
            stmt.executeUpdate();
            this.logger.debug("Unlinked Discord account for Minecraft " + String.valueOf(minecraftUuid));
        }
        catch (SQLException e) {
            this.logger.error("Error unlinking Discord account", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public UUID getLinkedMinecraftAccount(String discordId) {
        String sql = "SELECT minecraft_uuid FROM discord_links WHERE discord_id = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, discordId);
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return null;
            UUID uUID = UUID.fromString(rs.getString("minecraft_uuid"));
            return uUID;
        }
        catch (SQLException e) {
            this.logger.error("Error getting linked Minecraft account", e);
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public String getLinkedDiscordAccount(UUID minecraftUuid) {
        String sql = "SELECT discord_id FROM discord_links WHERE minecraft_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, minecraftUuid.toString());
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return null;
            String string = rs.getString("discord_id");
            return string;
        }
        catch (SQLException e) {
            this.logger.error("Error getting linked Discord account", e);
        }
        return null;
    }

    public void saveDiscordMessage(String discordId, String role, String content) {
        String sql = "INSERT INTO discord_conversations (discord_id, role, content, timestamp)\nVALUES (?, ?, ?, ?)\n";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, discordId);
            stmt.setString(2, role);
            stmt.setString(3, content);
            stmt.setLong(4, System.currentTimeMillis());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Error saving Discord message", e);
        }
    }

    public List<Conversation> getDiscordConversationHistory(String discordId, int limit) {
        ArrayList<Conversation> conversations = new ArrayList<Conversation>();
        String sql = "SELECT role, content, timestamp\nFROM discord_conversations\nWHERE discord_id = ?\nORDER BY timestamp DESC\nLIMIT ?\n";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, discordId);
            stmt.setInt(2, limit);
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                conversations.add(new Conversation(0, null, rs.getString("role"), rs.getString("content"), rs.getLong("timestamp")));
            }
        }
        catch (SQLException e) {
            this.logger.error("Error retrieving Discord conversation history", e);
        }
        return conversations;
    }

    public void deleteDiscordConversationHistory(String discordId) {
        String sql = "DELETE FROM discord_conversations WHERE discord_id = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, discordId);
            int deleted = stmt.executeUpdate();
            this.logger.debug("Deleted " + deleted + " Discord messages for user " + discordId);
        }
        catch (SQLException e) {
            this.logger.error("Error deleting Discord conversation history", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int getDiscordMessageCount(String discordId) {
        String sql = "SELECT COUNT(*) as count FROM discord_conversations WHERE discord_id = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, discordId);
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return 0;
            int n = rs.getInt("count");
            return n;
        }
        catch (SQLException e) {
            this.logger.error("Error getting Discord message count", e);
        }
        return 0;
    }

    public void logCommand(UUID playerUuid, String playerName, String command, boolean success, boolean dryRun) {
        String sql = "INSERT INTO command_logs (player_uuid, player_name, command, success, dry_run, timestamp) VALUES (?, ?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            stmt.setString(2, playerName);
            stmt.setString(3, command);
            stmt.setInt(4, success ? 1 : 0);
            stmt.setInt(5, dryRun ? 1 : 0);
            stmt.setLong(6, System.currentTimeMillis());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Error logging command execution", e);
        }
    }

    public void logModeration(UUID playerUuid, String playerName, String message, String severity, String action, String reason) {
        String sql = "INSERT INTO moderation_logs (player_uuid, player_name, message, severity, action, reason, timestamp) VALUES (?, ?, ?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            stmt.setString(2, playerName);
            stmt.setString(3, message);
            stmt.setString(4, severity);
            stmt.setString(5, action);
            stmt.setString(6, reason);
            stmt.setLong(7, System.currentTimeMillis());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Error logging moderation action", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public String getPlayerActiveBot(UUID playerUuid) {
        try {
            String sql = "SELECT active_bot FROM players WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                ResultSet rs = stmt.executeQuery();
                if (!rs.next()) return "default";
                String botName = rs.getString("active_bot");
                String string = botName != null ? botName : "default";
                return string;
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to get active bot for player: " + String.valueOf(playerUuid), e);
        }
        return "default";
    }

    public void setPlayerActiveBot(UUID playerUuid, String botName) {
        try {
            String sql = "UPDATE players SET active_bot = ? WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, botName);
                stmt.setString(2, playerUuid.toString());
                stmt.executeUpdate();
            }
            this.logger.debug("Set active bot for " + String.valueOf(playerUuid) + " to: " + botName);
        }
        catch (SQLException e) {
            this.logger.error("Failed to set active bot for player: " + String.valueOf(playerUuid), e);
        }
    }

    public void saveMessageForBot(UUID playerUuid, String botName, String role, String content) {
        try {
            String sql = "INSERT INTO conversations (player_uuid, bot_name, role, content, timestamp) VALUES (?, ?, ?, ?, ?)";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                stmt.setString(2, botName);
                stmt.setString(3, role);
                stmt.setString(4, content);
                stmt.setLong(5, System.currentTimeMillis());
                stmt.executeUpdate();
            }
            this.logger.debug("Saved " + role + " message for player " + String.valueOf(playerUuid) + " (bot: " + botName + ")");
        }
        catch (SQLException e) {
            this.logger.error("Failed to save conversation message for bot", e);
        }
    }

    public List<Conversation> getConversationHistoryForBot(UUID playerUuid, String botName, int limit) {
        ArrayList<Conversation> history = new ArrayList<Conversation>();
        try {
            String sql = "SELECT id, player_uuid, role, content, timestamp\nFROM conversations\nWHERE player_uuid = ? AND (bot_name = ? OR bot_name IS NULL)\nORDER BY timestamp DESC\nLIMIT ?\n";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                stmt.setString(2, botName);
                stmt.setInt(3, limit);
                ResultSet rs = stmt.executeQuery();
                while (rs.next()) {
                    history.add(new Conversation(rs.getInt("id"), UUID.fromString(rs.getString("player_uuid")), rs.getString("role"), rs.getString("content"), rs.getLong("timestamp")));
                }
            }
            Collections.reverse(history);
            this.logger.debug("Retrieved " + history.size() + " messages for " + String.valueOf(playerUuid) + " (bot: " + botName + ")");
        }
        catch (SQLException e) {
            this.logger.error("Failed to get conversation history for bot", e);
        }
        return history;
    }

    public void clearConversationForBot(UUID playerUuid, String botName) {
        try {
            String sql = botName == null ? "DELETE FROM conversations WHERE player_uuid = ?" : "DELETE FROM conversations WHERE player_uuid = ? AND bot_name = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                if (botName != null) {
                    stmt.setString(2, botName);
                }
                int deleted = stmt.executeUpdate();
                this.logger.info("Cleared " + deleted + " messages for " + String.valueOf(playerUuid) + (String)(botName != null ? " (bot: " + botName + ")" : " (all bots)"));
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to clear conversation for bot", e);
        }
    }

    public void createKnowledgeBase(KnowledgeBase kb) {
        String sql = "INSERT INTO knowledge_bases (id, name, description, collection, chunk_count, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, kb.getId());
            stmt.setString(2, kb.getName());
            stmt.setString(3, kb.getDescription());
            stmt.setString(4, kb.getCollection());
            stmt.setInt(5, kb.getChunkCount());
            stmt.setLong(6, kb.getCreatedAt());
            stmt.setLong(7, kb.getUpdatedAt());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Failed to create knowledge base", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public KnowledgeBase getKnowledgeBase(String kbId) {
        String sql = "SELECT * FROM knowledge_bases WHERE id = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, kbId);
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return null;
            KnowledgeBase kb = new KnowledgeBase(rs.getString("id"), rs.getString("name"), rs.getString("description"), rs.getString("collection"));
            kb.setChunkCount(rs.getInt("chunk_count"));
            kb.setCreatedAt(rs.getLong("created_at"));
            kb.setUpdatedAt(rs.getLong("updated_at"));
            KnowledgeBase knowledgeBase = kb;
            return knowledgeBase;
        }
        catch (SQLException e) {
            this.logger.error("Failed to get knowledge base", e);
        }
        return null;
    }

    public List<KnowledgeBase> getAllKnowledgeBases() {
        ArrayList<KnowledgeBase> kbs = new ArrayList<KnowledgeBase>();
        String sql = "SELECT * FROM knowledge_bases ORDER BY created_at DESC";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                KnowledgeBase kb = new KnowledgeBase(rs.getString("id"), rs.getString("name"), rs.getString("description"), rs.getString("collection"));
                kb.setChunkCount(rs.getInt("chunk_count"));
                kb.setCreatedAt(rs.getLong("created_at"));
                kb.setUpdatedAt(rs.getLong("updated_at"));
                kbs.add(kb);
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to get all knowledge bases", e);
        }
        return kbs;
    }

    public void updateKnowledgeBase(KnowledgeBase kb) {
        String sql = "UPDATE knowledge_bases SET name = ?, description = ?, chunk_count = ?, updated_at = ? WHERE id = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, kb.getName());
            stmt.setString(2, kb.getDescription());
            stmt.setInt(3, kb.getChunkCount());
            stmt.setLong(4, kb.getUpdatedAt());
            stmt.setString(5, kb.getId());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Failed to update knowledge base", e);
        }
    }

    public void deleteKnowledgeBase(String kbId) {
        String sql = "DELETE FROM knowledge_bases WHERE id = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, kbId);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Failed to delete knowledge base", e);
        }
    }

    public void createDocumentChunk(DocumentChunk chunk) {
        String sql = "INSERT INTO document_chunks (id, kb_id, source_file, chunk_index, content_preview, created_at) VALUES (?, ?, ?, ?, ?, ?)";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, chunk.getId());
            stmt.setString(2, chunk.getKbId());
            stmt.setString(3, chunk.getSourceFile());
            stmt.setInt(4, chunk.getChunkIndex());
            stmt.setString(5, chunk.getPreview());
            stmt.setLong(6, chunk.getCreatedAt());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Failed to create document chunk", e);
        }
    }

    public List<DocumentChunk> getDocumentChunksByKB(String kbId) {
        ArrayList<DocumentChunk> chunks = new ArrayList<DocumentChunk>();
        String sql = "SELECT * FROM document_chunks WHERE kb_id = ? ORDER BY source_file, chunk_index";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, kbId);
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                DocumentChunk chunk = new DocumentChunk(rs.getString("id"), rs.getString("kb_id"), rs.getString("source_file"), rs.getInt("chunk_index"), rs.getString("content_preview"));
                chunk.setCreatedAt(rs.getLong("created_at"));
                chunks.add(chunk);
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to get document chunks", e);
        }
        return chunks;
    }

    public void deleteDocumentChunksByKB(String kbId) {
        String sql = "DELETE FROM document_chunks WHERE kb_id = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, kbId);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Failed to delete document chunks", e);
        }
    }
}

