/*
 * Decompiled with CFR 0.152.
 */
package com.nexvor.dashboard.routes;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.nexvor.Nexvor;
import com.nexvor.ai.models.ChatRequest;
import com.nexvor.core.PluginTier;
import com.nexvor.dashboard.auth.AuthenticationManager;
import com.nexvor.dashboard.models.DashboardSession;
import com.nexvor.memory.DatabaseManager;
import com.nexvor.rag.models.KnowledgeBase;
import io.javalin.Javalin;
import io.javalin.http.Context;
import java.io.File;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.bukkit.Bukkit;
import org.bukkit.plugin.Plugin;

public class APIRoutes {
    private final Nexvor plugin;
    private final AuthenticationManager authManager;
    private final DatabaseManager databaseManager;

    public APIRoutes(Nexvor plugin, AuthenticationManager authManager) {
        this.plugin = plugin;
        this.authManager = authManager;
        this.databaseManager = plugin.getDatabaseManager();
    }

    public void register(Javalin app) {
        app.get("/api/docs", this::handleDocs);
        app.get("/api/stats", this::handleStats);
        app.get("/api/server", this::handleServerInfo);
        app.get("/api/conversations/recent", this::handleRecentConversations);
        app.get("/api/conversations/{uuid}", this::handlePlayerConversation);
        app.post("/api/ai/test", this::handleAITest);
        app.get("/api/ai/analytics", this::handleAIAnalytics);
        app.get("/api/kb/list", this::handleKBList);
        app.get("/api/kb/{id}/files", this::handleKBFiles);
        app.post("/api/kb/{id}/upload", this::handleKBUpload);
        app.delete("/api/kb/{id}/file/{filename}", this::handleKBDelete);
        app.get("/api/config", this::handleGetConfig);
        app.post("/api/config/reload", this::handleReloadConfig);
    }

    private DashboardSession requireAuth(Context ctx) {
        String authHeader = ctx.header("Authorization");
        if (authHeader == null || !authHeader.startsWith("Bearer ")) {
            ctx.status(401).json("{\"error\": \"Unauthorized\"}");
            return null;
        }
        String sessionToken = authHeader.substring(7);
        DashboardSession session = this.authManager.validateSession(sessionToken);
        if (session == null) {
            ctx.status(401).json("{\"error\": \"Invalid or expired session\"}");
            return null;
        }
        return session;
    }

    private void handleDocs(Context ctx) {
        JsonObject docs = new JsonObject();
        docs.addProperty("version", this.plugin.getDescription().getVersion());
        docs.addProperty("docs_url", "https://nexvor.dev/docs/api");
        JsonArray endpoints = new JsonArray();
        endpoints.add(this.createEndpoint("GET", "/api/stats", "read", "Get server statistics (players, TPS, memory)"));
        endpoints.add(this.createEndpoint("GET", "/api/server", "read", "Get server information (name, version, MOTD)"));
        endpoints.add(this.createEndpoint("GET", "/api/conversations/recent", "read", "Get recent AI conversations (use ?limit=N)"));
        endpoints.add(this.createEndpoint("GET", "/api/conversations/{uuid}", "read", "Get conversation history for a player"));
        endpoints.add(this.createEndpoint("POST", "/api/ai/test", "write", "Test AI with custom prompt"));
        endpoints.add(this.createEndpoint("GET", "/api/ai/analytics", "read", "Get AI usage analytics"));
        endpoints.add(this.createEndpoint("GET", "/api/kb/list", "read", "List knowledge bases (MAX tier)"));
        endpoints.add(this.createEndpoint("GET", "/api/kb/{id}/files", "read", "List files in knowledge base"));
        endpoints.add(this.createEndpoint("DELETE", "/api/kb/{id}/file/{filename}", "admin", "Delete file from knowledge base"));
        endpoints.add(this.createEndpoint("GET", "/api/config", "admin", "Get current configuration"));
        endpoints.add(this.createEndpoint("POST", "/api/config/reload", "admin", "Reload plugin configuration"));
        docs.add("endpoints", endpoints);
        JsonObject auth = new JsonObject();
        auth.addProperty("type", "Bearer Token");
        auth.addProperty("header", "Authorization: Bearer nvr_your_token_here");
        auth.addProperty("scopes", "read, write, admin");
        docs.add("authentication", auth);
        ctx.json(docs);
    }

    private JsonObject createEndpoint(String method2, String path, String scope, String description) {
        JsonObject endpoint = new JsonObject();
        endpoint.addProperty("method", method2);
        endpoint.addProperty("path", path);
        endpoint.addProperty("scope", scope);
        endpoint.addProperty("description", description);
        return endpoint;
    }

    private void handleStats(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        JsonObject stats = new JsonObject();
        stats.addProperty("online_players", Bukkit.getOnlinePlayers().size());
        stats.addProperty("max_players", Bukkit.getMaxPlayers());
        stats.addProperty("tps", Math.min(20.0, Bukkit.getTPS()[0]));
        stats.addProperty("plugin_version", this.plugin.getDescription().getVersion());
        stats.addProperty("tier", this.plugin.getActiveTier().name());
        Runtime runtime = Runtime.getRuntime();
        long usedMemory = (runtime.totalMemory() - runtime.freeMemory()) / 0x100000L;
        long maxMemory = runtime.maxMemory() / 0x100000L;
        stats.addProperty("memory_used_mb", usedMemory);
        stats.addProperty("memory_max_mb", maxMemory);
        ctx.json(stats);
    }

    private void handleServerInfo(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        JsonObject info = new JsonObject();
        info.addProperty("server_name", this.plugin.getConfig().getString("dashboard.ui.server_name", "Minecraft Server"));
        info.addProperty("server_version", Bukkit.getVersion());
        info.addProperty("bukkit_version", Bukkit.getBukkitVersion());
        info.addProperty("motd", Bukkit.getMotd());
        ctx.json(info);
    }

    private void handleKBList(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        if (!this.plugin.hasTier(PluginTier.MAX)) {
            ctx.status(403).json("{\"error\": \"Requires MAX tier\"}");
            return;
        }
        if (this.plugin.getKnowledgeBaseManager() == null) {
            ctx.status(503).json("{\"error\": \"Knowledge base manager not initialized\"}");
            return;
        }
        List<KnowledgeBase> kbs = this.plugin.getKnowledgeBaseManager().listKnowledgeBases();
        JsonArray array = new JsonArray();
        for (KnowledgeBase kb : kbs) {
            JsonObject obj = new JsonObject();
            obj.addProperty("id", kb.getId());
            obj.addProperty("name", kb.getName());
            obj.addProperty("description", kb.getDescription());
            obj.addProperty("collection", kb.getCollection());
            obj.addProperty("chunk_count", kb.getChunkCount());
            obj.addProperty("created_at", kb.getCreatedAt());
            obj.addProperty("updated_at", kb.getUpdatedAt());
            array.add(obj);
        }
        ctx.json(array);
    }

    private void handleGetConfig(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        if (!session.isAdmin()) {
            ctx.status(403).json("{\"error\": \"Admin access required\"}");
            return;
        }
        JsonObject config = new JsonObject();
        config.addProperty("ai_provider", this.plugin.getConfig().getString("api.provider"));
        config.addProperty("bot_name", this.plugin.getConfigManager().getBotName());
        config.addProperty("rate_limiting_enabled", this.plugin.getConfig().getBoolean("rate_limiting.enabled"));
        config.addProperty("language_filter_enabled", this.plugin.getConfig().getBoolean("language_filter.enabled"));
        ctx.json(config);
    }

    private void handleReloadConfig(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        if (!session.isAdmin()) {
            ctx.status(403).json("{\"error\": \"Admin access required\"}");
            return;
        }
        Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> this.plugin.reload());
        JsonObject response = new JsonObject();
        response.addProperty("success", true);
        response.addProperty("message", "Configuration reloaded successfully");
        ctx.json(response);
    }

    private void handleRecentConversations(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        int limit = ctx.queryParamAsClass("limit", Integer.class).getOrDefault(50);
        JsonArray conversations = new JsonArray();
        String sql = "SELECT player_uuid, player_name, message, response, timestamp FROM conversations ORDER BY timestamp DESC LIMIT ?";
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql);){
            stmt.setInt(1, limit);
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                JsonObject conv = new JsonObject();
                conv.addProperty("player_uuid", rs.getString("player_uuid"));
                conv.addProperty("player_name", rs.getString("player_name"));
                conv.addProperty("message", rs.getString("message"));
                conv.addProperty("response", rs.getString("response"));
                conv.addProperty("timestamp", rs.getLong("timestamp"));
                conversations.add(conv);
            }
        }
        catch (Exception e) {
            ctx.status(500).json("{\"error\": \"Database error\"}");
            return;
        }
        ctx.json(conversations);
    }

    private void handlePlayerConversation(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        String playerUUID = ctx.pathParam("uuid");
        JsonArray history = new JsonArray();
        String sql = "SELECT message, response, timestamp FROM conversations WHERE player_uuid = ? ORDER BY timestamp DESC LIMIT 100";
        try (Connection conn = this.databaseManager.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql);){
            stmt.setString(1, playerUUID);
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                JsonObject msg = new JsonObject();
                msg.addProperty("message", rs.getString("message"));
                msg.addProperty("response", rs.getString("response"));
                msg.addProperty("timestamp", rs.getLong("timestamp"));
                history.add(msg);
            }
        }
        catch (Exception e) {
            ctx.status(500).json("{\"error\": \"Database error\"}");
            return;
        }
        ctx.json(history);
    }

    private void handleAITest(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        Gson gson = new Gson();
        JsonObject request = gson.fromJson(ctx.body(), JsonObject.class);
        String prompt = request.get("prompt").getAsString();
        String systemPrompt = request.has("system_prompt") ? request.get("system_prompt").getAsString() : this.plugin.getConfigManager().getSystemPrompt();
        ChatRequest.Builder requestBuilder = ChatRequest.builder().model(this.plugin.getProviderManager().getPrimaryProvider().getDefaultModel()).addSystemMessage(systemPrompt).addUserMessage(prompt).temperature(0.7).maxTokens(300);
        ((CompletableFuture)this.plugin.getProviderManager().sendRequest(requestBuilder.build()).thenAccept(aiResponse -> {
            JsonObject result = new JsonObject();
            result.addProperty("response", aiResponse.getContent());
            result.addProperty("tokens", aiResponse.getTotalTokens());
            result.addProperty("model", aiResponse.getModel());
            ctx.json(result);
        })).exceptionally(error -> {
            ctx.status(500).json("{\"error\": \"AI request failed: " + error.getMessage() + "\"}");
            return null;
        });
    }

    private void handleAIAnalytics(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        JsonObject analytics = new JsonObject();
        String countSql = "SELECT COUNT(*) as total FROM conversations";
        long last24h = System.currentTimeMillis() - 86400000L;
        String recent24hSql = "SELECT COUNT(*) as count FROM conversations WHERE timestamp > ?";
        String topPlayersSql = "SELECT player_name, COUNT(*) as count FROM conversations GROUP BY player_uuid ORDER BY count DESC LIMIT 5";
        try (Connection conn = this.databaseManager.getConnection();){
            ResultSet rs;
            try (PreparedStatement stmt = conn.prepareStatement(countSql);){
                rs = stmt.executeQuery();
                if (rs.next()) {
                    analytics.addProperty("total_conversations", rs.getInt("total"));
                }
            }
            stmt = conn.prepareStatement(recent24hSql);
            try {
                stmt.setLong(1, last24h);
                rs = stmt.executeQuery();
                if (rs.next()) {
                    analytics.addProperty("requests_24h", rs.getInt("count"));
                }
            }
            finally {
                if (stmt != null) {
                    stmt.close();
                }
            }
            JsonArray topPlayers = new JsonArray();
            try (PreparedStatement stmt = conn.prepareStatement(topPlayersSql);){
                ResultSet rs2 = stmt.executeQuery();
                while (rs2.next()) {
                    JsonObject player = new JsonObject();
                    player.addProperty("name", rs2.getString("player_name"));
                    player.addProperty("requests", rs2.getInt("count"));
                    topPlayers.add(player);
                }
            }
            analytics.add("top_players", topPlayers);
        }
        catch (Exception e) {
            ctx.status(500).json("{\"error\": \"Database error\"}");
            return;
        }
        ctx.json(analytics);
    }

    private void handleKBFiles(Context ctx) {
        File[] fileList;
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        String kbId = ctx.pathParam("id");
        File kbDir = new File(this.plugin.getDataFolder(), "knowledge_base/" + kbId);
        JsonArray files = new JsonArray();
        if (kbDir.exists() && kbDir.isDirectory() && (fileList = kbDir.listFiles()) != null) {
            for (File file : fileList) {
                if (!file.isFile()) continue;
                JsonObject fileObj = new JsonObject();
                fileObj.addProperty("name", file.getName());
                fileObj.addProperty("size", file.length());
                fileObj.addProperty("modified", file.lastModified());
                files.add(fileObj);
            }
        }
        ctx.json(files);
    }

    private void handleKBUpload(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        ctx.status(501).json("{\"error\": \"File upload not yet implemented\"}");
    }

    private void handleKBDelete(Context ctx) {
        DashboardSession session = this.requireAuth(ctx);
        if (session == null) {
            return;
        }
        if (!session.isAdmin()) {
            ctx.status(403).json("{\"error\": \"Admin access required\"}");
            return;
        }
        String kbId = ctx.pathParam("id");
        String filename = ctx.pathParam("filename");
        File file = new File(this.plugin.getDataFolder(), "knowledge_base/" + kbId + "/" + filename);
        if (file.exists() && file.delete()) {
            ctx.json("{\"success\": true}");
        } else {
            ctx.status(404).json("{\"error\": \"File not found\"}");
        }
    }
}

