/*
 * Decompiled with CFR 0.152.
 */
package com.nexvor.dashboard.auth;

import com.nexvor.Nexvor;
import com.nexvor.core.NexvorLogger;
import com.nexvor.dashboard.auth.AccessCodeGenerator;
import com.nexvor.dashboard.models.AccessCode;
import com.nexvor.dashboard.models.DashboardSession;
import com.nexvor.memory.DatabaseManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;
import org.mindrot.jbcrypt.BCrypt;

public class AuthenticationManager {
    private final Nexvor plugin;
    private final NexvorLogger logger;
    private final DatabaseManager databaseManager;
    private final AccessCodeGenerator accessCodeGenerator;
    private final int sessionTimeout;

    public AuthenticationManager(Nexvor plugin) {
        this.plugin = plugin;
        this.logger = plugin.getNexvorLogger();
        this.databaseManager = plugin.getDatabaseManager();
        this.accessCodeGenerator = new AccessCodeGenerator(plugin);
        this.sessionTimeout = plugin.getConfig().getInt("dashboard.sessions.timeout_seconds", 3600);
    }

    public DashboardSession loginWithAccessCode(String code) {
        if (!this.plugin.getConfig().getBoolean("dashboard.auth.access_codes.enabled", true)) {
            this.logger.debug("Access code auth disabled");
            return null;
        }
        AccessCode accessCode = this.accessCodeGenerator.validate(code);
        if (accessCode == null) {
            return null;
        }
        return this.createSession(accessCode.getPlayerUuid(), accessCode.getPlayerName(), false);
    }

    public DashboardSession loginWithAdmin(String username, String password) {
        if (!this.plugin.getConfig().getBoolean("dashboard.auth.admin_enabled", true)) {
            this.logger.debug("Admin auth disabled");
            return null;
        }
        String configUsername = this.plugin.getConfig().getString("dashboard.auth.admin_username", "admin");
        String configPassword = this.plugin.getConfig().getString("dashboard.auth.admin_password");
        if (configPassword == null || configPassword.isEmpty()) {
            this.logger.warn("Admin password not set in config.yml!");
            return null;
        }
        if (!username.equals(configUsername)) {
            this.logger.debug("Admin login failed: invalid username");
            return null;
        }
        boolean passwordValid = configPassword.startsWith("$2a$") || configPassword.startsWith("$2b$") ? BCrypt.checkpw(password, configPassword) : password.equals(configPassword);
        if (!passwordValid) {
            this.logger.debug("Admin login failed: invalid password");
            return null;
        }
        this.logger.info("Admin logged in via dashboard");
        return this.createSession("admin", "Admin", true);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public DashboardSession loginWithCredentials(String username, String password) {
        if (!this.plugin.getConfig().getBoolean("dashboard.auth.user_accounts.enabled", false)) {
            this.logger.debug("User account auth disabled");
            return null;
        }
        try {
            String sql = "SELECT * FROM dashboard_users WHERE player_name = ?";
            try (PreparedStatement stmt = this.databaseManager.getConnection().prepareStatement(sql);){
                stmt.setString(1, username);
                ResultSet rs = stmt.executeQuery();
                if (!rs.next()) return null;
                String passwordHash = rs.getString("password_hash");
                String playerUuid = rs.getString("player_uuid");
                if (BCrypt.checkpw(password, passwordHash)) {
                    this.updateLastLogin(playerUuid);
                    this.logger.info(username + " logged in via dashboard");
                    DashboardSession dashboardSession = this.createSession(playerUuid, username, false);
                    return dashboardSession;
                }
                this.logger.debug("User login failed: invalid password for " + username);
                return null;
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to login with credentials", e);
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean registerUser(UUID playerUuid, String playerName, String password) {
        if (!this.plugin.getConfig().getBoolean("dashboard.auth.user_accounts.enabled", false)) {
            return false;
        }
        int minLength = this.plugin.getConfig().getInt("dashboard.auth.user_accounts.min_password_length", 8);
        if (password.length() < minLength) {
            return false;
        }
        try {
            String checkSql = "SELECT * FROM dashboard_users WHERE player_uuid = ?";
            try (PreparedStatement stmt = this.databaseManager.getConnection().prepareStatement(checkSql);){
                stmt.setString(1, playerUuid.toString());
                ResultSet rs = stmt.executeQuery();
                if (rs.next()) {
                    this.logger.debug("User already registered: " + playerName);
                    boolean bl = false;
                    return bl;
                }
            }
            String passwordHash = BCrypt.hashpw(password, BCrypt.gensalt());
            String insertSql = "INSERT INTO dashboard_users (player_uuid, player_name, password_hash, created_at) VALUES (?, ?, ?, ?)";
            try (PreparedStatement stmt2 = this.databaseManager.getConnection().prepareStatement(insertSql);){
                stmt2.setString(1, playerUuid.toString());
                stmt2.setString(2, playerName);
                stmt2.setString(3, passwordHash);
                stmt2.setLong(4, System.currentTimeMillis());
                stmt2.executeUpdate();
            }
            this.logger.info("Dashboard account registered for " + playerName);
            return true;
        }
        catch (SQLException e) {
            this.logger.error("Failed to register user", e);
            return false;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public DashboardSession validateSession(String sessionToken) {
        try {
            String sql = "SELECT * FROM dashboard_sessions WHERE session_token = ?";
            try (PreparedStatement stmt = this.databaseManager.getConnection().prepareStatement(sql);){
                stmt.setString(1, sessionToken);
                ResultSet rs = stmt.executeQuery();
                if (!rs.next()) return null;
                DashboardSession session = new DashboardSession(rs.getString("id"), rs.getString("user_uuid"), rs.getString("user_name"), rs.getString("session_token"), rs.getLong("created_at"), rs.getLong("expires_at"), rs.getLong("last_active"), rs.getString("user_uuid").equals("admin"));
                if (session.isValid()) {
                    this.updateSessionLastActive(sessionToken);
                    DashboardSession dashboardSession = session;
                    return dashboardSession;
                }
                this.deleteSession(sessionToken);
                return null;
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to validate session", e);
        }
        return null;
    }

    private DashboardSession createSession(String userUuid, String userName, boolean isAdmin) {
        String sessionId = UUID.randomUUID().toString();
        String sessionToken = UUID.randomUUID().toString();
        long expiresAt = System.currentTimeMillis() + (long)this.sessionTimeout * 1000L;
        DashboardSession session = new DashboardSession(sessionId, userUuid, userName, sessionToken, expiresAt, isAdmin);
        try {
            String sql = "INSERT INTO dashboard_sessions (id, user_uuid, user_name, session_token, created_at, expires_at, last_active) VALUES (?, ?, ?, ?, ?, ?, ?)";
            try (PreparedStatement stmt = this.databaseManager.getConnection().prepareStatement(sql);){
                stmt.setString(1, sessionId);
                stmt.setString(2, userUuid);
                stmt.setString(3, userName);
                stmt.setString(4, sessionToken);
                stmt.setLong(5, session.getCreatedAt());
                stmt.setLong(6, expiresAt);
                stmt.setLong(7, session.getLastActive());
                stmt.executeUpdate();
            }
            this.logger.debug("Created dashboard session for " + userName);
            return session;
        }
        catch (SQLException e) {
            this.logger.error("Failed to create session", e);
            return null;
        }
    }

    private void updateSessionLastActive(String sessionToken) throws SQLException {
        String sql = "UPDATE dashboard_sessions SET last_active = ? WHERE session_token = ?";
        try (PreparedStatement stmt = this.databaseManager.getConnection().prepareStatement(sql);){
            stmt.setLong(1, System.currentTimeMillis());
            stmt.setString(2, sessionToken);
            stmt.executeUpdate();
        }
    }

    private void deleteSession(String sessionToken) throws SQLException {
        String sql = "DELETE FROM dashboard_sessions WHERE session_token = ?";
        try (PreparedStatement stmt = this.databaseManager.getConnection().prepareStatement(sql);){
            stmt.setString(1, sessionToken);
            stmt.executeUpdate();
        }
    }

    private void updateLastLogin(String playerUuid) throws SQLException {
        String sql = "UPDATE dashboard_users SET last_login = ? WHERE player_uuid = ?";
        try (PreparedStatement stmt = this.databaseManager.getConnection().prepareStatement(sql);){
            stmt.setLong(1, System.currentTimeMillis());
            stmt.setString(2, playerUuid);
            stmt.executeUpdate();
        }
    }

    public void cleanupSessions() {
        try {
            long now = System.currentTimeMillis();
            String sql = "DELETE FROM dashboard_sessions WHERE expires_at < ?";
            try (PreparedStatement stmt = this.databaseManager.getConnection().prepareStatement(sql);){
                stmt.setLong(1, now);
                int deleted = stmt.executeUpdate();
                if (deleted > 0) {
                    this.logger.debug("Cleaned up " + deleted + " expired dashboard sessions");
                }
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to cleanup sessions", e);
        }
    }

    public AccessCodeGenerator getAccessCodeGenerator() {
        return this.accessCodeGenerator;
    }
}

