/*
 * Decompiled with CFR 0.152.
 */
package com.nexvor.memory;

import com.nexvor.Nexvor;
import com.nexvor.core.NexvorLogger;
import com.nexvor.events.EventType;
import com.nexvor.memory.models.Conversation;
import com.nexvor.memory.models.PlayerData;
import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public class DatabaseManager {
    private final Nexvor plugin;
    private final NexvorLogger logger;
    private final File databaseFile;
    private Connection connection;

    public DatabaseManager(Nexvor plugin) {
        this.plugin = plugin;
        this.logger = plugin.getNexvorLogger();
        this.databaseFile = new File(plugin.getDataFolder(), "nexvor.db");
    }

    public void initialize() {
        try {
            if (!this.plugin.getDataFolder().exists()) {
                this.plugin.getDataFolder().mkdirs();
            }
            Class.forName("org.sqlite.JDBC");
            String url = "jdbc:sqlite:" + this.databaseFile.getAbsolutePath();
            this.connection = DriverManager.getConnection(url);
            this.logger.info("Database connected: " + this.databaseFile.getAbsolutePath());
            this.createTables();
        }
        catch (ClassNotFoundException e) {
            this.logger.severe("SQLite JDBC driver not found! This should not happen.");
            e.printStackTrace();
        }
        catch (SQLException e) {
            this.logger.severe("Failed to initialize database!");
            e.printStackTrace();
        }
    }

    private void createTables() throws SQLException {
        String createPlayersTable = "CREATE TABLE IF NOT EXISTS players (\n    uuid TEXT PRIMARY KEY,\n    username TEXT NOT NULL,\n    first_seen INTEGER NOT NULL,\n    last_seen INTEGER NOT NULL,\n    opted_out INTEGER DEFAULT 0\n)\n";
        String createConversationsTable = "CREATE TABLE IF NOT EXISTS conversations (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    player_uuid TEXT NOT NULL,\n    role TEXT NOT NULL,\n    content TEXT NOT NULL,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
        String createConversationsIndex = "CREATE INDEX IF NOT EXISTS idx_conversations_player_timestamp\nON conversations(player_uuid, timestamp DESC)\n";
        String createGroupsTable = "CREATE TABLE IF NOT EXISTS conversation_groups (\n    group_id TEXT PRIMARY KEY,\n    group_name TEXT NOT NULL,\n    owner_uuid TEXT NOT NULL,\n    created_at INTEGER NOT NULL,\n    active INTEGER DEFAULT 1,\n    FOREIGN KEY (owner_uuid) REFERENCES players(uuid)\n)\n";
        String createGroupMembersTable = "CREATE TABLE IF NOT EXISTS group_members (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    group_id TEXT NOT NULL,\n    player_uuid TEXT NOT NULL,\n    joined_at INTEGER NOT NULL,\n    FOREIGN KEY (group_id) REFERENCES conversation_groups(group_id),\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid),\n    UNIQUE(group_id, player_uuid)\n)\n";
        String createGroupMessagesTable = "CREATE TABLE IF NOT EXISTS group_messages (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    group_id TEXT NOT NULL,\n    player_uuid TEXT,\n    role TEXT NOT NULL,\n    content TEXT NOT NULL,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (group_id) REFERENCES conversation_groups(group_id),\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
        String createGroupMessagesIndex = "CREATE INDEX IF NOT EXISTS idx_group_messages_group_timestamp\nON group_messages(group_id, timestamp DESC)\n";
        String createGlobalMessagesTable = "CREATE TABLE IF NOT EXISTS global_messages (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    conversation_id TEXT NOT NULL,\n    player_uuid TEXT,\n    role TEXT NOT NULL,\n    content TEXT NOT NULL,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
        String createGlobalMessagesIndex = "CREATE INDEX IF NOT EXISTS idx_global_messages_conversation_timestamp\nON global_messages(conversation_id, timestamp DESC)\n";
        String createProximityMessagesTable = "CREATE TABLE IF NOT EXISTS proximity_messages (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    group_id TEXT NOT NULL,\n    player_uuid TEXT,\n    role TEXT NOT NULL,\n    content TEXT NOT NULL,\n    timestamp INTEGER NOT NULL,\n    FOREIGN KEY (player_uuid) REFERENCES players(uuid)\n)\n";
        String createProximityMessagesIndex = "CREATE INDEX IF NOT EXISTS idx_proximity_messages_group_timestamp\nON proximity_messages(group_id, timestamp DESC)\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createPlayersTable);
            stmt.execute(createConversationsTable);
            stmt.execute(createConversationsIndex);
            stmt.execute(createGroupsTable);
            stmt.execute(createGroupMembersTable);
            stmt.execute(createGroupMessagesTable);
            stmt.execute(createGroupMessagesIndex);
            stmt.execute(createGlobalMessagesTable);
            stmt.execute(createGlobalMessagesIndex);
            stmt.execute(createProximityMessagesTable);
            stmt.execute(createProximityMessagesIndex);
            String createEventPreferencesTable = "CREATE TABLE IF NOT EXISTS event_preferences (\n    player_uuid TEXT NOT NULL,\n    event_type TEXT NOT NULL,\n    enabled INTEGER NOT NULL DEFAULT 1,\n    PRIMARY KEY (player_uuid, event_type)\n)\n";
            stmt.execute(createEventPreferencesTable);
            this.logger.info("Database tables created/verified");
        }
    }

    public void close() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                this.connection.close();
                this.logger.info("Database connection closed");
            }
        }
        catch (SQLException e) {
            this.logger.error("Error closing database connection", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public PlayerData getOrCreatePlayer(UUID uuid, String username) {
        try {
            String selectSql = "SELECT * FROM players WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(selectSql);){
                stmt.setString(1, uuid.toString());
                ResultSet rs = stmt.executeQuery();
                if (rs.next()) {
                    this.updatePlayer(uuid, username);
                    PlayerData playerData = new PlayerData(uuid, rs.getString("username"), rs.getLong("first_seen"), rs.getLong("last_seen"), rs.getInt("opted_out") == 1);
                    return playerData;
                }
            }
            long now = System.currentTimeMillis();
            String insertSql = "INSERT INTO players (uuid, username, first_seen, last_seen, opted_out) VALUES (?, ?, ?, ?, 0)";
            try (PreparedStatement stmt2 = this.connection.prepareStatement(insertSql);){
                stmt2.setString(1, uuid.toString());
                stmt2.setString(2, username);
                stmt2.setLong(3, now);
                stmt2.setLong(4, now);
                stmt2.executeUpdate();
            }
            this.logger.debug("Created new player record: " + username + " (" + String.valueOf(uuid) + ")");
            return new PlayerData(uuid, username, now, now, false);
        }
        catch (SQLException e) {
            this.logger.error("Failed to get/create player: " + username, e);
            return null;
        }
    }

    private void updatePlayer(UUID uuid, String username) {
        try {
            String sql = "UPDATE players SET username = ?, last_seen = ? WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, username);
                stmt.setLong(2, System.currentTimeMillis());
                stmt.setString(3, uuid.toString());
                stmt.executeUpdate();
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to update player: " + username, e);
        }
    }

    public void saveMessage(UUID playerUuid, String role, String content) {
        try {
            String sql = "INSERT INTO conversations (player_uuid, role, content, timestamp) VALUES (?, ?, ?, ?)";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                stmt.setString(2, role);
                stmt.setString(3, content);
                stmt.setLong(4, System.currentTimeMillis());
                stmt.executeUpdate();
            }
            this.logger.debug("Saved " + role + " message for player " + String.valueOf(playerUuid));
        }
        catch (SQLException e) {
            this.logger.error("Failed to save conversation message", e);
        }
    }

    public List<Conversation> getConversationHistory(UUID playerUuid, int limit) {
        ArrayList<Conversation> history = new ArrayList<Conversation>();
        try {
            String sql = "SELECT id, player_uuid, role, content, timestamp\nFROM conversations\nWHERE player_uuid = ?\nORDER BY timestamp DESC\nLIMIT ?\n";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                stmt.setInt(2, limit);
                ResultSet rs = stmt.executeQuery();
                while (rs.next()) {
                    history.add(new Conversation(rs.getInt("id"), UUID.fromString(rs.getString("player_uuid")), rs.getString("role"), rs.getString("content"), rs.getLong("timestamp")));
                }
            }
            ArrayList<Conversation> reversed = new ArrayList<Conversation>();
            for (int i = history.size() - 1; i >= 0; --i) {
                reversed.add((Conversation)history.get(i));
            }
            return reversed;
        }
        catch (SQLException e) {
            this.logger.error("Failed to get conversation history", e);
            return new ArrayList<Conversation>();
        }
    }

    public void clearConversationHistory(UUID playerUuid) {
        try {
            String sql = "DELETE FROM conversations WHERE player_uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                int deleted = stmt.executeUpdate();
                this.logger.info("Cleared " + deleted + " messages for player " + String.valueOf(playerUuid));
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to clear conversation history", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int getMessageCount(UUID playerUuid) {
        try {
            String sql = "SELECT COUNT(*) FROM conversations WHERE player_uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                ResultSet rs = stmt.executeQuery();
                if (!rs.next()) return 0;
                int n = rs.getInt(1);
                return n;
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to get message count", e);
        }
        return 0;
    }

    public void setOptOut(UUID playerUuid, boolean optedOut) {
        try {
            String sql = "UPDATE players SET opted_out = ? WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setInt(1, optedOut ? 1 : 0);
                stmt.setString(2, playerUuid.toString());
                stmt.executeUpdate();
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to set opt-out status", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean isOptedOut(UUID playerUuid) {
        try {
            String sql = "SELECT opted_out FROM players WHERE uuid = ?";
            try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
                stmt.setString(1, playerUuid.toString());
                ResultSet rs = stmt.executeQuery();
                if (!rs.next()) return false;
                boolean bl = rs.getInt("opted_out") == 1;
                return bl;
            }
        }
        catch (SQLException e) {
            this.logger.error("Failed to check opt-out status", e);
        }
        return false;
    }

    public Connection getConnection() {
        return this.connection;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Boolean getEventPreference(UUID playerUuid, EventType eventType) {
        String sql = "SELECT enabled FROM event_preferences WHERE player_uuid = ? AND event_type = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            stmt.setString(2, eventType.getConfigKey());
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return null;
            Boolean bl = rs.getInt("enabled") == 1;
            return bl;
        }
        catch (SQLException e) {
            this.logger.error("Error getting event preference", e);
        }
        return null;
    }

    public void setEventPreference(UUID playerUuid, EventType eventType, boolean enabled) {
        String sql = "INSERT INTO event_preferences (player_uuid, event_type, enabled)\nVALUES (?, ?, ?)\nON CONFLICT(player_uuid, event_type)\nDO UPDATE SET enabled = excluded.enabled\n";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            stmt.setString(2, eventType.getConfigKey());
            stmt.setInt(3, enabled ? 1 : 0);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Error setting event preference", e);
        }
    }

    public Map<String, Boolean> getAllEventPreferences(UUID playerUuid) {
        HashMap<String, Boolean> preferences = new HashMap<String, Boolean>();
        String sql = "SELECT event_type, enabled FROM event_preferences WHERE player_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            ResultSet rs = stmt.executeQuery();
            while (rs.next()) {
                preferences.put(rs.getString("event_type"), rs.getInt("enabled") == 1);
            }
        }
        catch (SQLException e) {
            this.logger.error("Error getting all event preferences", e);
        }
        return preferences;
    }

    public void clearEventPreferences(UUID playerUuid) {
        String sql = "DELETE FROM event_preferences WHERE player_uuid = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, playerUuid.toString());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.logger.error("Error clearing event preferences", e);
        }
    }
}

